// Leaflet用CanvasOverlay
// 機能: ImageOverlayと同様に、地図上の矩形領域にCanvasを描画する
// 作成者: Joe Masumura (ASH Corporation.)
// 利用例:
//   var size = [2000, 1245]; // 横, 縦
//   var range = [[-1.399, -3.272], [5.710, 8.148]]; // 左下[y,x], 右上[y,x]
//   var can = L.canvasOverlay(size, range, {
//     opacity: 0.9,
//     zIndex: 20
//   });
//   can.addTo(map);

L.CanvasOverlay = L.ImageOverlay.extend({
	initialize: function (size, range, options) { // コンストラクタ
		this._canSize = [size[0], size[1]]; // x, y
		this._range = [
			[range[0][0], range[0][1]],
			[range[1][0], range[1][1]]]; // [y, x], [y, x]
		this._mapSize = [
			this._range[1][1] - this._range[0][1], // x
			this._range[1][0] - this._range[0][0]]; // y
		this._bounds = L.latLngBounds(this._range);

		L.setOptions(this, options);
	},

	_initImage: function() { // Canvasの初期登録
		if (this.options.canvas) {
			this._image = this.options.canvas;
		} else {
			this._image = document.createElement('canvas');
		}
		this._image.width = this._canSize[0];
		this._image.height = this._canSize[1];

		L.DomUtil.addClass(this._image, 'leaflet-canvas-layer');
		if (this._zoomAnimated) { L.DomUtil.addClass(this._image, 'leaflet-zoom-animated'); }
		if (this.options.className) { L.DomUtil.addClass(this._image, this.options.className); }

		this._image.onselectstart = false;
		this._image.onmousemove = false;

		this.getPane().appendChild(this._image);
	},

	getContext: function () { // 描画オブジェクトの取得
		if (this._image.getContext) {
			return this._image.getContext('2d');
		} else {
			return null;
		}
	},

	getScale: function () { // キャンバスのスケールの取得
		return (this._mapSize[0] + this._mapSize[1])
			 / (this._canSize[0] + this._canSize[1]);
	},

	can2map: function (canPos) { // キャンバス座標(x,y)からマップ座標(lat,lng)に変換
		var x, y;
		if (canPos instanceof Array) {
			x = canPos[0];
			y = canPos[1];
		} else {
			x = canPos.x;
			y = canPos.y;
		}

		var latlng = new Object();
		latlng.lat = this._range[1][0] - y / this._canSize[1] * this._mapSize[1];
		latlng.lng = this._range[0][1] + x / this._canSize[0] * this._mapSize[0];
		return (latlng);
	},

	map2can: function (mapPos) { // マップ座標(lat,lng)からキャンバス座標(x,y)に変換
		var x, y;
		if (mapPos instanceof Array) {
			y = mapPos[0];
			x = mapPos[1];
		} else {
			y = mapPos.lat;
			x = mapPos.lng;
		}

		var pos = new Object();
		pos.x = (x - this._range[0][1]) / this._mapSize[0] * this._canSize[0];
		pos.y = (this._range[1][0] - y) / this._mapSize[1] * this._canSize[1];
		return (pos);
	}
});

L.canvasOverlay = function(size, range, options) { // 小文字で始まるショートカットの定義
    return new L.CanvasOverlay(size, range, options);
}
